// Session Timeout Management
// Handles 15-minute inactivity timeout across all tabs/windows

class SessionTimeoutManager {
    constructor() {
        this.timeoutMinutes = 15;
        this.timeoutMs = this.timeoutMinutes * 60 * 1000; // 15 minutes in milliseconds
        this.warningMinutes = 2; // Show warning 2 minutes before timeout
        this.warningMs = this.warningMinutes * 60 * 1000;
        this.lastActivity = Date.now();
        this.warningShown = false;
        this.timeoutId = null;
        this.warningId = null;
        
        this.init();
    }
    
    init() {
        // Set up activity listeners
        this.setupActivityListeners();
        
        // Start the timeout timer
        this.startTimeoutTimer();
        
        // Check for timeout on page load
        this.checkForTimeout();
        
        // Set up periodic checks
        setInterval(() => this.checkForTimeout(), 30000); // Check every 30 seconds
        
        // Listen for storage events (for cross-tab communication)
        window.addEventListener('storage', (e) => {
            if (e.key === 'session_timeout' && e.newValue === 'logout') {
                this.forceLogout('Session expired due to inactivity');
            }
        });
    }
    
    setupActivityListeners() {
        const events = ['mousedown', 'mousemove', 'keypress', 'scroll', 'touchstart', 'click'];
        
        events.forEach(event => {
            document.addEventListener(event, () => {
                this.resetActivity();
            }, true);
        });
        
        // Also listen for focus events
        window.addEventListener('focus', () => {
            this.resetActivity();
        });
    }
    
    resetActivity() {
        this.lastActivity = Date.now();
        this.warningShown = false;
        
        // Clear existing timers
        if (this.timeoutId) {
            clearTimeout(this.timeoutId);
        }
        if (this.warningId) {
            clearTimeout(this.warningId);
        }
        
        // Restart timers
        this.startTimeoutTimer();
    }
    
    startTimeoutTimer() {
        const timeSinceActivity = Date.now() - this.lastActivity;
        const remainingTime = this.timeoutMs - timeSinceActivity;
        const warningTime = this.warningMs - timeSinceActivity;
        
        // Set warning timer
        if (warningTime > 0) {
            this.warningId = setTimeout(() => {
                this.showWarning();
            }, warningTime);
        }
        
        // Set timeout timer
        if (remainingTime > 0) {
            this.timeoutId = setTimeout(() => {
                this.handleTimeout();
            }, remainingTime);
        } else {
            // Already timed out
            this.handleTimeout();
        }
    }
    
    showWarning() {
        if (this.warningShown) return;
        
        this.warningShown = true;
        
        // Create warning modal
        const warningModal = document.createElement('div');
        warningModal.id = 'session-warning-modal';
        warningModal.innerHTML = `
            <div class="modal fade" tabindex="-1" style="display: block;">
                <div class="modal-dialog modal-dialog-centered">
                    <div class="modal-content">
                        <div class="modal-header bg-warning">
                            <h5 class="modal-title">
                                <i class="fas fa-exclamation-triangle"></i> Session Timeout Warning
                            </h5>
                        </div>
                        <div class="modal-body">
                            <p>Your session will expire in ${this.warningMinutes} minutes due to inactivity.</p>
                            <p>Click anywhere or press any key to continue your session.</p>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-primary" onclick="sessionManager.extendSession()">
                                Continue Session
                            </button>
                            <button type="button" class="btn btn-secondary" onclick="sessionManager.logout()">
                                Logout Now
                            </button>
                        </div>
                    </div>
                </div>
            </div>
            <div class="modal-backdrop fade show"></div>
        `;
        
        document.body.appendChild(warningModal);
        
        // Auto-hide warning after 30 seconds
        setTimeout(() => {
            this.hideWarning();
        }, 30000);
    }
    
    hideWarning() {
        const warningModal = document.getElementById('session-warning-modal');
        if (warningModal) {
            warningModal.remove();
        }
        this.warningShown = false;
    }
    
    extendSession() {
        this.hideWarning();
        this.resetActivity();
        
        // Send heartbeat to server
        this.sendHeartbeat();
    }
    
    async sendHeartbeat() {
        try {
            const response = await fetch('../api/heartbeat.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    timestamp: Date.now()
                })
            });
            
            if (!response.ok) {
                throw new Error('Heartbeat failed');
            }
        } catch (error) {
            console.error('Heartbeat error:', error);
            // If heartbeat fails, force logout
            this.forceLogout('Connection lost');
        }
    }
    
    handleTimeout() {
        // Notify other tabs
        localStorage.setItem('session_timeout', 'logout');
        
        // Force logout
        this.forceLogout('Session expired due to inactivity');
    }
    
    forceLogout(message = 'Session expired') {
        // Clear any existing modals
        this.hideWarning();
        
        // Show logout message
        this.showLogoutMessage(message);
        
        // Redirect to login page after 3 seconds
        setTimeout(() => {
            window.location.href = '../auth/login.php?timeout=1';
        }, 3000);
    }
    
    showLogoutMessage(message) {
        const logoutModal = document.createElement('div');
        logoutModal.id = 'session-logout-modal';
        logoutModal.innerHTML = `
            <div class="modal fade" tabindex="-1" style="display: block;">
                <div class="modal-dialog modal-dialog-centered">
                    <div class="modal-content">
                        <div class="modal-header bg-danger text-white">
                            <h5 class="modal-title">
                                <i class="fas fa-sign-out-alt"></i> Session Expired
                            </h5>
                        </div>
                        <div class="modal-body">
                            <p>${message}</p>
                            <p>You will be redirected to the login page.</p>
                        </div>
                    </div>
                </div>
            </div>
            <div class="modal-backdrop fade show"></div>
        `;
        
        document.body.appendChild(logoutModal);
    }
    
    logout() {
        // Clear any existing modals
        this.hideWarning();
        
        // Redirect to logout
        window.location.href = '../auth/logout.php';
    }
    
    checkForTimeout() {
        const timeSinceActivity = Date.now() - this.lastActivity;
        
        if (timeSinceActivity >= this.timeoutMs) {
            this.handleTimeout();
        } else if (timeSinceActivity >= this.warningMs && !this.warningShown) {
            this.showWarning();
        }
    }
}

// Initialize session timeout manager when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    window.sessionManager = new SessionTimeoutManager();
});

// Also initialize immediately if DOM is already loaded
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', function() {
        window.sessionManager = new SessionTimeoutManager();
    });
} else {
    window.sessionManager = new SessionTimeoutManager();
} 